import asyncio
import json

from py_pli.pylib import VUnits, Measurements, send_msg
from py_pli.pylib import EndPointType, URPCFunctions
from pylog.pylogger import PyLogger
from virtualunits.HAL import HAL

import time


async def upper_heating(setpoint, enable=0):
    if enable == 0:
        await VUnits.instance.hal.upper_Heating.disable()
        pass
    await VUnits.instance.hal.upper_Heating.InitializeDevice()
    await VUnits.instance.hal.upper_Heating.set_target_temperature(setpoint)
    await VUnits.instance.hal.upper_Heating.enable()




async def lower_heating(setpoint, enable=0):
    if enable == 0:
        await VUnits.instance.hal.lower_Heating.disable()
        pass
    await VUnits.instance.hal.lower_Heating.InitializeDevice()
    await VUnits.instance.hal.lower_Heating.set_target_temperature(setpoint)
    await VUnits.instance.hal.lower_Heating.enable()


async def avu_heating(setpoint, enable=0):
    if enable == 0:
        await VUnits.instance.hal.lower_Heating.disable()
        pass
    await VUnits.instance.hal.airVentilation_Heating.InitializeDevice()
    await VUnits.instance.hal.airVentilation_Heating.set_target_temperature(setpoint)
    await VUnits.instance.hal.airVentilation_Heating.enable()

async def avu_cooling(setpoint, enable=0):
    if enable == 0:
        await VUnits.instance.hal.lower_Heating.disable()
        pass
    await VUnits.instance.hal.airVentilation_Cooling.InitializeDevice()
    await VUnits.instance.hal.airVentilation_Cooling.set_target_temperature(setpoint)
    await VUnits.instance.hal.airVentilation_Cooling.enable()

async def reset_mainboard():
    msg = {'result': f'reset mainboard...'}
    await send_msg(json.dumps(msg))

    mainboard = 0x0020
    node = URPCFunctions.instance.endPointsDic.get(mainboard)
    PyLogger.logger.info(f"Reset Mainboard")
    try:
        await node.Reset()
        await asyncio.sleep(5)
    except:
        PyLogger.logger.info(f"Reset Mainboard not possible!")
    finally:
        PyLogger.logger.info(f"Reset Mainboard done")

async def start_firmware():
    msg = {'result': f'start firmware...'}
    await send_msg(json.dumps(msg))

    mainboard = 0x0020
    node = URPCFunctions.instance.endPointsDic.get(mainboard)
    await node.StartFirmware()
    await asyncio.sleep(10)
    PyLogger.logger.info(f"Startup Mainboard firmware done!")

async def get_temp_value_from_sensor(sensor_number):
    mainboard = 0x0020
    node = URPCFunctions.instance.endPointsDic.get(mainboard)
    temp_val = round((await node.GetAnalogInput(sensor_number))[0] * 256, 2)
    return temp_val

async def wait_until_reach_setpoint(temp_sensor, setpoint):
    start_time = time.time()

    wait = 1
    while (wait):
        threshold_time = time.time()
        temperature = await get_temp_value_from_sensor(temp_sensor)
        if temperature >= setpoint:
            msg = {'result': f'Setpoint has been reached after {round(threshold_time - start_time, 2)}s'}
            await send_msg(json.dumps(msg))
            break
        await asyncio.sleep(1)
        msg = {'result': f'wait until setpoint has been reached...'}
        await send_msg(json.dumps(msg))

async def wait_until_duration_of_measurement(duration=3600):
    msg = {'result': f'measurement started'}
    await send_msg(json.dumps(msg))

    time_to_sleep = duration
    while (time_to_sleep):
        time_to_sleep -= 1
        msg = {'result': f'measurement ends in {time_to_sleep}s'}
        await send_msg(json.dumps(msg))
        await asyncio.sleep(1)

async def two_degrees_above_ambient_temp(temp_sensor, duration=3600):

    await reset_mainboard()
    await start_firmware()

    ambient_sensor = 4
    two_degrees = 2
    ambient_temp = await get_temp_value_from_sensor(ambient_sensor )
    setpoint = ambient_temp + two_degrees
    msg = {'result': f'ambient temperature = {ambient_temp} °C setpoint = {setpoint}°C'}
    await send_msg(json.dumps(msg))


    await upper_heating(setpoint, enable=1)
    await lower_heating(setpoint, enable=1)
    await avu_heating(setpoint, enable=1)
    msg = {'result': f'initialized devices...'}
    await send_msg(json.dumps(msg))
    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await upper_heating(setpoint, enable=0)
    await lower_heating(setpoint, enable=0)
    await avu_heating(setpoint, enable=0)
    PyLogger.logger.info(f"Finished, disable all temp_control")
    return f"two_degrees_above_ambient_temp done, all units disabled"

async def heat_with_foil_to_setpoint(temp_sensor, setpoint, duration=3600):
    await reset_mainboard()
    await start_firmware()

    await upper_heating(setpoint, enable=1)
    await lower_heating(setpoint, enable=1)

    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await upper_heating(setpoint, enable=0)
    await lower_heating(setpoint, enable=0)
    return f"heat_with_foil_to_setpoint done, all units disabled"


async def heat_with_foil_and_avu_to_setpoint(temp_sensor, setpoint, duration=3600):
    await reset_mainboard()
    await start_firmware()

    await upper_heating(setpoint, enable=1)
    await lower_heating(setpoint, enable=1)
    await avu_heating(setpoint, enable=1)

    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await upper_heating(setpoint, enable=0)
    await lower_heating(setpoint, enable=0)
    await avu_heating(setpoint, enable=0)
    return f"heat_with_foil_and_avu_to_setpoint done, all units disabled"


async def heat_with_upper_heating_to_setpoint(temp_sensor, setpoint, duration=3600):
    await reset_mainboard()
    await start_firmware()

    await upper_heating(setpoint, enable=1)

    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await upper_heating(setpoint, enable=0)
    return f"heat_with_upper_heating_to_setpoint done, all units disabled"


async def heat_with_lower_heating_to_setpoint(temp_sensor, setpoint, duration=3600):
    await reset_mainboard()
    await start_firmware()

    await lower_heating(setpoint, enable=1)

    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await lower_heating(setpoint, enable=0)
    return f"heat_with_lower_heating_to_setpoint done, all units disabled"


async def heat_with_avu_to_setpoint(temp_sensor, setpoint, duration=3600):
    await reset_mainboard()
    await start_firmware()

    await avu_heating(setpoint, enable=1)

    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await avu_heating(setpoint, enable=0)
    return f"heat_with_avu_to_setpoint done, all units disabled"


async def cool_with_avu_to_setpoint(temp_sensor, setpoint, duration=3600):
    await reset_mainboard()
    await start_firmware()

    await avu_cooling(setpoint, enable=1)

    await asyncio.sleep(10)

    await wait_until_reach_setpoint(temp_sensor=temp_sensor, setpoint=setpoint)
    await wait_until_duration_of_measurement(duration)

    await avu_cooling(setpoint, enable=0)
    return f"cool_with_avu_to_setpoint done, all units disabled"

async def condensation_prevention():
    msg = {'result': f'sorry, not available yet'}
    await send_msg(json.dumps(msg))